<?php

namespace App\Http\Controllers;

use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class LanguageController extends Controller
{
    public function index(Request $request)
    {
        $collection = Language::latest()->get();
        return view('language.index', compact('collection'));
    }


    public function create()
    {
        return view('language.create');
    }


    public function store(Request $request)
    {
        $this->validation($request);
        try {
            $data = $request->only('name', 'iso', 'status');
            if ($request->hasFile('icon')) {
                $data['icon'] = $this->fileUploader($request->file('icon'), 'language',80,80);
            }
            $language  = Language::create($data);
            if ($language){
                $this->updateLang($language->iso);
            }
            successAlert('Created successfully');
            return redirect()->route('language.index');
        } catch (\Exception $e) {
            errorAlert($e->getMessage());
            return redirect()->back();
        }
    }


    public function edit(Language $language)
    {
        return view('language.edit', compact('language'));
    }


    public function update(Request $request, Language $language)
    {
        $this->validation($request, $language->id);
        try {
            $data = $request->only('name', 'iso', 'status');
            if ($request->hasFile('icon')) {
                $data['icon'] = $this->fileUploader($request->file('icon'), 'language',80,80, $language->icon);
            }
            $this->updateLang($request->iso, $language->iso);
            $language->update($data);
            successAlert('Updated successfully');
            return redirect()->route('language.index');
        } catch (\Exception $e) {
            errorAlert($e->getMessage());
            return redirect()->back();
        }
    }

    public function delete(Language $language)
    {
        try {
            if (File::exists(resource_path('lang/'.$language->iso))){
                File::deleteDirectory(resource_path('lang/'.$language->iso));
            }
            $language->delete();
            session()->put('local','en');
            successAlert('Deleted successfully');
            return redirect()->route('language.index');
        } catch (\Exception $e) {
            errorAlert($e->getMessage());
            return redirect()->back();
        }
    }


    public function updateTerms(Request $request, Language $language)
    {
        $directory = resource_path('lang/'.$language->iso);
        $jsonFiles = glob($directory . '/*.json');
        $modules = [];

        foreach ($jsonFiles as $jsonFile) {
            $modules[] = pathinfo($jsonFile, PATHINFO_FILENAME);
        }

        $jsonString = [];

        if ($request->has('module')){
            $path = resource_path('lang/' . $language->iso.'/');
            $strings = file_get_contents($path . $request->input('module','common').'.json');
            $jsonString = json_decode($strings, true);
        }


       if ($request->method() == 'POST') {
           try {
               $strings = [];
               foreach ($request->str_key as $index =>  $key) {
                   $strings[$key] = $request->str_value[$index];
               }
               $module = strtolower($request->module);
               $file = base_path("resources/lang/{$language->iso}/{$module}.json");
               updateContent($file,$strings);
               successAlert('Updated successfully');
               return redirect()->back();
           } catch (\Exception $e) {
               errorAlert($e->getMessage());
               return redirect()->back();
           }
        }
        return view('language.terms', compact('language', 'modules', 'jsonString'));
    }


    protected function updateLang($newLang, $oldLang = null)
    {
        try {
            if ($oldLang){
                if (File::exists(resource_path('lang/'.$oldLang))){
                    File::deleteDirectory(resource_path('lang/'.$oldLang));
                }
            }
            if (File::exists(resource_path('lang/'.$newLang))){
                File::deleteDirectory(resource_path('lang/'.$newLang));
            }
            File::copyDirectory(resource_path('lang/en'), resource_path('lang/'.strtolower($newLang)));
            return true;
        }catch (\Exception $exception){
            throw $exception;
        }
    }


    private function validation($request, $id = null)
    {
        $arr = [
            'name' => 'required',
            'iso' => 'required|string|unique:languages,iso,'.$id,
            'status' => 'required',
        ];
        return $request->validate($arr);
    }

}
